////////////////////////////////////////////////////////////////////////////////
// HIDsample										Create Date	2015/11/12
//				for PIC16F1455						Last Update	2018/04/06
// file name :  main.c										Saka Softwares
// language  :  XC8 (v1.45)
////////////////////////////////////////////////////////////////////////////////
// HID bootloadr (pic16f1455x_Family) 及び usbhiddev.c の動作サンプルです。
// 動作確認は、書き込み Microchips製 USB Bootloader v2.15
// 及び HIDConsole Version 1.5で行っています。
// usbhiddev.cは、PIC18と共用できるように変更しています。
// main.cを入れ替えるだけで PIC18にも対応します。
////////////////////////////////////////////////////////////////////////////////
// Include Files ///////////////////////////////////////////////////////////////
#include <xc.h>
#include <stdint.h>
#include <stdbool.h>

#include "usbhiddev.h"

// Configuration ///////////////////////////////////////////////////////////////
#define USE_INTERNAL_OSC			// 内部発振でUSBを使う時
//#define USE_HID_BOOTLOADER			// Loadables にプートローダーを設定する時
// PIC16のコンフィグ生成で同じ設定をしても mpasmとxc8で違う値なります。
// mpasmは、0x3FFFが基準で xc8は、0xFFFFが基準。その為コードが不一致とみなされます。
// 書き込めば同じなのですが…
#ifndef USE_HID_BOOTLOADER
#if defined(USE_INTERNAL_OSC)
// CONFIG1
#pragma config FOSC = INTOSC		// Oscillator Selection Bits (INTOSC oscillator: I/O function on CLKIN pin)
#pragma config WDTE = SWDTEN		// Watchdog Timer Enable (WDT controlled by the SWDTEN bit in the WDTCON register)
#pragma config PWRTE = ON			// Power-up Timer Enable (PWRT enabled)
#pragma config MCLRE = OFF			// MCLR Pin Function Select (MCLR/VPP pin function is digital input)
#pragma config CP = OFF				// Flash Program Memory Code Protection (Program memory code protection is disabled)
#pragma config BOREN = ON			// Brown-out Reset Enable (Brown-out Reset enabled)
#pragma config CLKOUTEN = OFF		// Clock Out Enable (CLKOUT function is disabled. I/O or oscillator function on the CLKOUT pin)
#pragma config IESO = OFF			// Internal/External Switchover Mode (Internal/External Switchover Mode is disabled)
#pragma config FCMEN = OFF			// Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor is disabled)
// CONFIG2
#pragma config WRT = OFF			// Flash Memory Self-Write Protection (Write protection off)
#pragma config CPUDIV = NOCLKDIV	// CPU System Clock Selection Bit (NO CPU system divide)
#pragma config USBLSCLK = 48MHz		// USB Low SPeed Clock Selection bit (System clock expects 48 MHz, FS/LS USB CLKENs divide-by is set to 8.)
#pragma config PLLMULT = 3x			// PLL Multipler Selection Bit (3x Output Frequency Selected)
#pragma config PLLEN = ENABLED		// PLL Enable Bit (3x or 4x PLL Enabled)
#pragma config STVREN = ON			// Stack Overflow/Underflow Reset Enable (Stack Overflow or Underflow will cause a Reset)
#pragma config BORV = LO			// Brown-out Reset Voltage Selection (Brown-out Reset Voltage (Vbor), low trip point selected.)
#pragma config LPBOR = OFF			// Low-Power Brown Out Reset (Low-Power BOR is disabled)
#pragma config LVP = OFF			// Low-Voltage Programming Enable (High-voltage on MCLR/VPP must be used for programming)
#else
// CONFIG1 (crystal:12MHz)
#pragma config FOSC = HS			// Oscillator Selection Bits (HS Oscillator, High-speed crystal/resonator connected between OSC1 and OSC2 pins)
#pragma config WDTE = SWDTEN		// Watchdog Timer Enable (WDT controlled by the SWDTEN bit in the WDTCON register)
#pragma config PWRTE = ON			// Power-up Timer Enable (PWRT enabled)
#pragma config MCLRE = OFF			// MCLR Pin Function Select (MCLR/VPP pin function is digital input)
#pragma config CP = OFF				// Flash Program Memory Code Protection (Program memory code protection is disabled)
#pragma config BOREN = ON			// Brown-out Reset Enable (Brown-out Reset enabled)
#pragma config CLKOUTEN = OFF		// Clock Out Enable (CLKOUT function is disabled. I/O or oscillator function on the CLKOUT pin)
#pragma config IESO = OFF			// Internal/External Switchover Mode (Internal/External Switchover Mode is disabled)
#pragma config FCMEN = OFF			// Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor is disabled)
// CONFIG2
#pragma config WRT = OFF			// Flash Memory Self-Write Protection (Write protection off)
#pragma config CPUDIV = NOCLKDIV	// CPU System Clock Selection Bit (NO CPU system divide)
#pragma config USBLSCLK = 48MHz		// USB Low SPeed Clock Selection bit (System clock expects 48 MHz, FS/LS USB CLKENs divide-by is set to 8.)
#pragma config PLLMULT = 4x			// PLL Multipler Selection Bit (4x Output Frequency Selected)
#pragma config PLLEN = ENABLED		// PLL Enable Bit (3x or 4x PLL Enabled)
#pragma config STVREN = ON			// Stack Overflow/Underflow Reset Enable (Stack Overflow or Underflow will cause a Reset)
#pragma config BORV = LO			// Brown-out Reset Voltage Selection (Brown-out Reset Voltage (Vbor), low trip point selected.)
#pragma config LPBOR = OFF			// Low-Power Brown Out Reset (Low-Power BOR is disabled)
#pragma config LVP = OFF			// Low-Voltage Programming Enable (High-voltage on MCLR/VPP must be used for programming)
#endif
#endif

// Version info ////////////////////////////////////////////////////////////////
//注. 下記 VersionWord は、XC8 (v1.36)で旨く書き込めない場合があります。
//    HID bootloadrを使わない場合コメントアウトしてください。
const uint16_t VersionWord @ 0x0502 = 0x0100;	// BCD表現のバージョン No.
#define	SOFTWEAR_VERSION	"sample_hid16f145x (XC8) Ver1.0"

// Hardware Settings ///////////////////////////////////////////////////////////
// LEDs
#define LED_ON		1
#define LED_OFF		0
#define LED1		LATAbits.LATA5
#define LED1Tris	TRISAbits.TRISA5

// SWITCH
#define SW_ON		0
#define SW_OFF		1
#define sw2			PORTAbits.RA3

// Port入出力
#define PORTOUT		0
#define PORTIN		1
// Variable ////////////////////////////////////////////////////////////////////
#define DATA_SIZE	64			// EP1Outと同じか大きくしてください。
uint8_t rcv_dat[DATA_SIZE];		// 受信データ

union {
	uint8_t Val;				//フラグ値
	struct {
		unsigned VER:1;			//バージョン送信フラグ
		unsigned HELP:1;		//ヘルプ送信フラグ
		unsigned ECHO:1;		//エコーフラグ
		unsigned :6;
	}bits;
} FLAG;							// 各種フラグ

// Fixed number ////////////////////////////////////////////////////////////////
// ヘルプデータ
const uint8_t *HELP[] = {
	"Help",
	" v : Version",
	" e : Echoback ON",
	" E : Echoback OFF",
	" h : Help",
	" Boot : Bootloder mode",
	" ",
	""
};

// Internal Function ///////////////////////////////////////////////////////////
void main();
void interrupt Isr();
void ProcessIO();
void bootloader(uint8_t* pbuf);
void MainUsbStatus();

////////////////////////////////////////////////////////////////////////////////
// メイン処理部
// 引  数	: 無し
// 戻り値	: 無し
////////////////////////////////////////////////////////////////////////////////
void main() {
	CLRWDT();
	INTCON  = 0x00;					// 割り込み禁止
	STKPTR  = 0x00;					// スタックポインタ設定
	WDTCON  = 0x17;					// ウォッチドックタイマ起動（2s）
#if defined(USE_INTERNAL_OSC)
	OSCCON = 0xFC;					//HFINTOSC @ 16MHz, 3X PLL, PLL enabled
	ACTCON = 0x90;					//Active clock tuning enabled for USB
#endif
	FLAG.Val = 0x00;				// フラグクリア
#if defined(ENABLE_USB_LED_BLINK_STATUS)
	LED1 = LED_OFF;
	LED1Tris = PORTOUT;
#endif
	USBDeviceInit();				// 割り込み対応USBの初期化
	INTCONbits.GIE  = 1	;			// 割り込み許可
	INTCONbits.PEIE = 1	;			// 周辺割り込み許可
	while (1) {
		CLRWDT();
		bootloader(rcv_dat);		// ブートローダー移行処理を行います。
		USBDeviceTasks();			// USBのバス接続切断処理を行います。
		MainUsbStatus();			// USBフラグを見てLEDの点灯処理を行います。
		ProcessIO();				// 通信処理
    }    
}

////////////////////////////////////////////////////////////////////////////////
// USB割り込み処理部処理部
// 引  数	: 無し
// 戻り値	: 無し
// USBのプラグ&プレイ関連の処理を行っています。必須です。 
////////////////////////////////////////////////////////////////////////////////
void interrupt Isr()
{
	// USB割り込み処理
	USBInterruptTasks();
}

////////////////////////////////////////////////////////////////////////////////
// プロセスＩＯ処理部
// 引  数	: 無し
// 戻り値	: 無し
// USB関連の処理を一括で行っています。
// HID受信した場合フラグを立てて各処理を行い、HID送信が可能になるまで待ち
// 結果を返す仕様にしています。
// 1回の受信または送信は64byteです。HIDのリポートディスクリプタで設定しています。
// 送信処理(HidTxRamReport)は、前の送信が終わるまで待ちます。
// 複数送る場合ウォッチドックのタイムアウトに気を付けてください。
////////////////////////////////////////////////////////////////////////////////
void ProcessIO()
{
	uint8_t len;
	if ((len = HidRxReport(rcv_dat, DATA_SIZE))  < RCV_RECEING) {
		// エコーの送信 /////////////////////////////////////////////////////////
		if (!FLAG.bits.ECHO) HidTxRamReport(rcv_dat, len);
		switch (rcv_dat[0]) {
			// バージョン情報の送信 /////////////////////////////////////////////
			case 'v' : {
				HidTxRomReport(SOFTWEAR_VERSION, sizeof(SOFTWEAR_VERSION));
				break;
			}
			// エコーバックのON /////////////////////////////////////////////////
			case 'e' : {
				FLAG.bits.ECHO = 0;
				break;
			}
			// エコーバックのOFF ////////////////////////////////////////////////
			case 'E' : {
				FLAG.bits.ECHO = 1;
				break;
			}
			// ヘルプの送信 /////////////////////////////////////////////////////
			case 'h' : {
				uint8_t cnt = 0;
				while ( *HELP[cnt] != '\0') {
					HidTxRomReport(HELP[cnt++], 22);
				}
				break;
			}
			default	 : break;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
// ブートローダー処理部
// 引  数	: 受信データバッファポインタ
// 戻り値	: 無し
// sw2または受信コマンドによりHIDブートローダーへ移行します。
// HID Bootloader (pic16f145x_Family)に対応しています。
////////////////////////////////////////////////////////////////////////////////
void bootloader(uint8_t* pbuf)
{
	if (!sw2 || (pbuf != 0 && *(pbuf+0) == 'B'
		&& *(pbuf+1) == 'o' && *(pbuf+2) == 'o' && *(pbuf+3) == 't')) {
		INTCONbits.GIE  = 0;				// 割り込み禁止
		INTCONbits.PEIE = 0	;				// 周辺割り込み許可
		LATA  = 0;							// ポートAクリア
		LATC  = 0;							// ポートCクリア
		TRISA = 0;							// ポートA初期化
		TRISC = 0;							// ポートC初期化
		uint16_t cnt = 0xFFF;
		while (cnt--) { 
			CLRWDT();
			UCON = 0;						// ウェイト付きUSB停止
		}
		asm("movlp 0x00");
		asm("goto  0x001C");				// bootloader Entryへ
	}
}

////////////////////////////////////////////////////////////////////////////////
// USBステータスによる各処理部
// 引  数	: 無し
// 戻り値	: 無し
// LED表示処理
// USBに未接続時は消灯 USB接続処理中は点灯 USBの通信中は点滅します。
//* サスペンド処理
// サスペンド時にスリープします。
//* その前後にユーザー処理を入れます。
////////////////////////////////////////////////////////////////////////////////
void MainUsbStatus()
{
#ifdef ENABLE_USB_LED_BLINK_STATUS
	// USB ステータス表示処理 ///////////////////////////////////////////////////
	static uint16_t led_count = 0;
	led_count--;
	if(led_count == 0) {
		led_count = 0x3000;
		if (USBGetDeviceState() == DETACHED_STATE) {
			LED1 = LED_OFF;				// 非接続	(消灯)
		}
		else if (USBGetDeviceState() < CONFIGURED_STATE) {
			LED1 = (unsigned)!LED1;		// 接続中	(点滅)
		}
		else {
			LED1 = LED_ON;				// 通信中	(点灯)
		}
	}
#endif
	// サスペンド処理 ///////////////////////////////////////////////////////////
	if (USBGetDeviceState() == CONFIGURED_STATE && USBIsSuspended() ) {
	// メインのサスペンド移行処理部
#ifdef ENABLE_USB_LED_BLINK_STATUS
		LED1 = LED_OFF;					// 通信中断	(消灯)
#endif
		// サスペンド中何もせずスリープする
		while (!USBIsActive() && !STATUSbits.nTO) {
			SLEEP();
		}
		// メインのウェイクアップ処理部
#ifdef ENABLE_USB_LED_BLINK_STATUS
		LED1 = LED_ON;					// 通信開始 (点灯)
#endif
	}
}
// EOF main.c //////////////////////////////////////////////////////////////////
